package com.iqsim.xiqs;

import javax.swing.*;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.*;
import java.net.UnknownHostException;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.iqsim.xiqs.utils.ClientSocketHandler;
import com.iqsim.xiqs.utils.ComplexFilter;
import com.iqsim.xiqs.utils.FilterToXMLAdder;
import com.iqsim.xiqs.utils.JavaFileToXML;
import com.iqsim.xiqs.utils.RequestSender;
import com.iqsim.xiqs.utils.SimpleFilter;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;

import com.iqsim.xiqs.interfaces.ServerResultListener;

public class FileUploadGUI extends JFrame implements ServerResultListener {

	// Generated Serial version UID
	private static final long serialVersionUID = -5974513049103332752L;

	// GUI components
	private JTextField loginField;
	private JPasswordField passwordField;
	private JTextField ipField;
	private JTextField portField;
	private DefaultListModel<File> fileListModel;
	private JList<File> fileList;
	private JTextArea xmlContentArea;
	private JTextArea responseArea;
	private JCheckBox autoLoginCheckBox;

	// Configuration properties (loaded from configuration file)
	private Properties config;

	// Client socket handler for server communication (setted global for the class for asynchroneous events)
	private ClientSocketHandler clientSocketHandler;

	// Configuration file name (Java standarts use .properties files)
	private static final String CONFIG_FILE = "config.properties";

	// Constructor to initialize the GUI and events
	public FileUploadGUI() {
		// Load configuration from file, we create new Properties object, then the loadConfig method load properties from file
		config = new Properties();
		loadConfig();

		// Set up the main frame
		setTitle("XIQS API Tester Java Version");
		setSize(1024, 768);
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setLocationRelativeTo(null);

		// Initialize GUI components
		loginField = new JTextField(20);
		passwordField = new JPasswordField(20);
		ipField = new JTextField(20);
		portField = new JTextField(20);
		xmlContentArea = new JTextArea(10, 50);
		responseArea = new JTextArea(10, 50);
		responseArea.setEditable(false);
		autoLoginCheckBox = new JCheckBox("Auto Login");


		//Button to load command files, two choices, xml format or java class files
		JButton xmlfileButton = new JButton("Select a .XML file");
		JButton javaFileButton = new JButton("Select a .java file");
		JButton sendButton = new JButton("Send to server");

		fileListModel = new DefaultListModel<>();
		fileList = new JList<>(fileListModel);
		fileList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		fileList.addListSelectionListener(e -> loadSelectedFileContent());

		// Set up the layout using GridBagLayout
		JPanel panel = new JPanel(new GridBagLayout());
		GridBagConstraints c = new GridBagConstraints();

		c.insets = new Insets(5, 5, 5, 5);

		// Add components to the panel
		c.fill = GridBagConstraints.HORIZONTAL;
		c.gridx = 0;
		c.gridy = 0;
		c.anchor = GridBagConstraints.WEST;
		panel.add(autoLoginCheckBox, c);

		c.fill = GridBagConstraints.HORIZONTAL;
		c.gridx = 0;
		c.gridy = 0;
		panel.add(new JLabel("Login:"), c);

		c.gridx = 1;
		c.gridy = 0;
		c.weightx = 1.0;
		panel.add(loginField, c);

		c.gridx = 0;
		c.gridy = 1;
		c.weightx = 0;
		panel.add(new JLabel("Password:"), c);

		c.gridx = 1;
		c.gridy = 1;
		c.weightx = 1.0;
		panel.add(passwordField, c);

		c.gridx = 0;
		c.gridy = 2;
		c.weightx = 0;
		panel.add(new JLabel("IP Address:"), c);

		c.gridx = 1;
		c.gridy = 2;
		c.weightx = 1.0;
		panel.add(ipField, c);

		c.gridx = 0;
		c.gridy = 3;
		c.weightx = 0;
		panel.add(new JLabel("Port:"), c);

		c.gridx = 1;
		c.gridy = 3;
		c.weightx = 1.0;
		panel.add(portField, c);

		c.gridx = 0;
		c.gridy = 4;
		c.gridwidth = 2;
		c.weightx = 1.0;
		c.fill = GridBagConstraints.NONE;
		panel.add(autoLoginCheckBox, c);

		c.gridx = 0;
		c.gridy = 5;
		c.gridwidth = 2;
		c.weightx = 1.0;
		c.fill = GridBagConstraints.NONE;
		panel.add(xmlfileButton, c);

		c.gridx = 0;
		c.gridy = 6;
		c.gridwidth = 1;
		panel.add(javaFileButton, c);

		c.gridx = 0;
		c.gridy = 7;
		c.gridwidth = 2;
		c.fill = GridBagConstraints.BOTH;
		c.weighty = 0.5;
		panel.add(new JScrollPane(fileList), c);

		c.gridx = 0;
		c.gridy = 8;
		c.gridwidth = 2;
		c.weighty = 1.0;
		panel.add(new JScrollPane(xmlContentArea), c);

		// Right-click menu for XML Request filter
		JPopupMenu popupMenuRequestFilter = new JPopupMenu();

		// Simple filter option
		JMenuItem simpleFilterOption = new JMenuItem("Add simple filter example");
		simpleFilterOption.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				xmlContentArea.setText(FilterToXMLAdder.addFilter(xmlContentArea, SimpleFilter.sampleString()));
			}
		});

		// Complex filter option
		JMenuItem complexFilterOption = new JMenuItem("Add complex filter example");
		complexFilterOption.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				xmlContentArea.setText(FilterToXMLAdder.addFilter(xmlContentArea, ComplexFilter.sampleString()));
			}
		});

		// Add filter options to the popup menu
		popupMenuRequestFilter.add(complexFilterOption);
		popupMenuRequestFilter.add(simpleFilterOption);

		// Add mouse listener for showing popup menu on XML content area
		xmlContentArea.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (e.isPopupTrigger()) {
					popupMenuRequestFilter.show(e.getComponent(), e.getX(), e.getY());
				}
			}
		});

		// Add send button to the panel
		c.gridx = 0;
		c.gridy = 9;
		c.gridwidth = 2;
		c.weighty = 0;
		panel.add(sendButton, c);

		// Add response area to the panel
		c.gridx = 0;
		c.gridy = 10;
		c.gridwidth = 2;
		c.weighty = 1.0;
		panel.add(new JScrollPane(responseArea), c);

		// Right-click menu for XML Response format
		JPopupMenu popupMenuResponseFormat = new JPopupMenu();

		// Erase response option
		JMenuItem eraseResponse = new JMenuItem("Erase");
		eraseResponse.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				responseArea.setText("");
			}
		});

		// Add erase response option to the popup menu
		popupMenuResponseFormat.add(eraseResponse);

		// Add mouse listener for showing popup menu on response area
		responseArea.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (e.isPopupTrigger()) {
					popupMenuResponseFormat.show(e.getComponent(), e.getX(), e.getY());
				}
			}
		});

		// Add the panel to the main frame
		add(panel);

		// Fill the fields with configuration values
		fillFieldsFromConfig();

		// Action listener for XML file button
		xmlfileButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				JFileChooser fileChooser = new JFileChooser();
				fileChooser.setMultiSelectionEnabled(true);
				int returnValue = fileChooser.showOpenDialog(null);
				if (returnValue == JFileChooser.APPROVE_OPTION) {
					File[] files = fileChooser.getSelectedFiles();
					for (File file : files) {
						fileListModel.addElement(file);
					}
				}
			}
		});

		// Action listener for Java file button
		javaFileButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				JFileChooser fileChooser = new JFileChooser();
				fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
				int returnValue = fileChooser.showOpenDialog(null);
				if (returnValue == JFileChooser.APPROVE_OPTION) {
					File selectedFile = fileChooser.getSelectedFile();

					try {
						// Convert Java file to XML to be able to test java class without compiling or parsing compiled class
						String transformedJavaCode = JavaFileToXML.generateXMLFromFile(selectedFile.getAbsolutePath());
						xmlContentArea.setText(transformedJavaCode);

					} catch (IOException | ParserConfigurationException | TransformerException ex) {
						showError(ex);
					}

				}
			}
		});

		// Action listener for Send button
		sendButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				String login = loginField.getText();
				String password = new String(passwordField.getPassword());
				String ip = ipField.getText();
				int port = Integer.parseInt(portField.getText());

				// Update configuration with current values, at each command send, the connections informations are saved
				updateConfig(login, password, ip, port, autoLoginCheckBox.isSelected());

				if (null != clientSocketHandler) {
					try {
						clientSocketHandler.initSocket(ip, port);
						Thread socketThread = new Thread(clientSocketHandler);
						socketThread.start();
					} catch (IOException e1) {
						showError(e1);
					}
				}

				//If autologin is enabled, login command is sent at each test. It's recommanded to keep ON to avoid sending a login request at each test
				if (autoLoginCheckBox.isSelected()) {
					// Create login request XML, hard coded, but can be loaded from file if needed
					String loginRequest = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" +
							"<command name=\"loginXIQS\">" +
							"  <param type=\"string\">" + login + "</param>" +
							"  <param type=\"string\">" + password + "</param>" +
							"</command>";
					clientSocketHandler.sendRequest(loginRequest);
				}

				// Send request to the server
				String request = xmlContentArea.getText();
				clientSocketHandler.sendRequest(request);
			}
		});

		try {
			// Initialize client socket handler and set listener, the listener is setted here because 
			// it must be the global gui class and not the ActionListener used for button. The context is different
			clientSocketHandler = new ClientSocketHandler();
			clientSocketHandler.setListener(this); // Server responses listener registered in the thread
		} catch (NumberFormatException ex) {
			this.showError(ex);
		}
	}

	// Load content of the selected file into the XML content area
	private void loadSelectedFileContent() {
		File selectedFile = fileList.getSelectedValue();
		if (selectedFile != null) {
			try {
				BufferedReader reader = new BufferedReader(new FileReader(selectedFile));
				xmlContentArea.read(reader, null);
				reader.close();
			} catch (IOException ex) {
				ex.printStackTrace();
				this.showError(ex);
			}
		}
	}

	// Load configuration from the properties file
	private void loadConfig() {
		try (InputStream input = new FileInputStream(CONFIG_FILE)) {
			config.load(input);
		} catch (IOException ex) {
			this.showError(ex);
		}
	}

	// Fill GUI fields with configuration values
	private void fillFieldsFromConfig() {
		loginField.setText(config.getProperty("login", ""));
		passwordField.setText(config.getProperty("password", ""));
		ipField.setText(config.getProperty("ip", ""));
		portField.setText(config.getProperty("port", ""));
		autoLoginCheckBox.setSelected(Boolean.parseBoolean(config.getProperty("autoLogin", "false")));
	}

	// Update configuration with current values and save to file
	private void updateConfig(String login, String password, String ip, int port, boolean autoLogin) {
		config.setProperty("login", login);
		config.setProperty("password", password);
		config.setProperty("ip", ip);
		config.setProperty("port", String.valueOf(port));
		config.setProperty("autoLogin", String.valueOf(autoLogin));

		try (OutputStream output = new FileOutputStream(CONFIG_FILE)) {
			config.store(output, null);
		} catch (IOException ex) {
			this.showError(ex);
		}
	}

	// Show error message dialog
	private void showError(Exception e) {
		JOptionPane.showMessageDialog(null, "An error occurred: " + e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
	}

	// Handle server responses
	@Override
	public void onServerResponse(String response) {
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				// Update response area with the server response
				responseArea.append(response + "\n");
			}
		});
	}

	// Main method to run the application
	public static void main(String[] args) {
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				// Show the main frame
				new FileUploadGUI().setVisible(true);
			}
		});
	}
}
