package com.iqsim.xiqs.utils;

import java.io.*;
import java.net.*;
import java.util.concurrent.*;
import java.util.concurrent.locks.*;

import com.iqsim.xiqs.interfaces.ServerResultListener;

//Thread class to send server request and retrieve response
public class ClientSocketHandler implements Runnable {
    private Socket socket; // Socket object to send request and retrieve response
    private OutputStream out; // Stream to send request
    private InputStream in; // Stream to retrieve response
    private final Lock lock = new ReentrantLock(); // Lock to ensure there is no thread conflicts
    private volatile boolean running = true; // Indicate if we are waiting for server response. Volatile to ensure we have the last variable state at each use
    private ServerResultListener listener; // Listener wich use server's response (In this sample program, it is FileUploadGUI)
    
    //Default Constructor, mandatory to setlistener on ActionEvent without socket
    public ClientSocketHandler() {
    	
    }

    //Constructor that initialize socket
    public ClientSocketHandler(String serverAddress, int port) throws IOException {
        this.socket = new Socket(serverAddress, port);
        this.out = socket.getOutputStream();
        this.in = socket.getInputStream();
    }
    
 
    //Thread to wait for server response
    @Override
    public void run() {

        //Buffer initialisation
    	 byte[] buffer = new byte[4096];
         int bytesRead;
         System.out.println("Waiting for server response...");
         
         try {
            //Running is an indicator that indicates we want to read response. but is socket is closed by server we can't
            // and we read bytes until it's finished
             while (running && !socket.isClosed() && (bytesRead = in.read(buffer)) != -1) {
                 String serverResponse = new String(buffer, 0, bytesRead);

                 //Only for debug, print response in console
                 System.out.println("Server: " + serverResponse);
                 
                 //Check if our listener (FileUploadGUI) is not null. Then we get it response
                 if (listener != null) {
                     listener.onServerResponse(serverResponse);
                 }
                 
             }

             //We stop process manualy or the server closed the socket
             System.out.println("Stopped listening to server.");
         } catch (IOException e) {
             if (running) {
                 e.printStackTrace();
             }
         } finally {
            // If process finished, close socket
             close();
         }
    } 

    //Method to send request to server, called by listener (FileUploadGUI)
    public void sendRequest(String request) {    
        //Lock resource to avoid thread conflict, One and only one thread at time can perform a request 
        lock.lock();
        try {
            //Write command content in the socket output stream
        	out.write(request.getBytes());
            //Ensure the request is send immediatly, can prevent infinite lock
            out.flush();

            //Only for debug, can be removed
        	System.out.println(request);
        } catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} finally {
            // Free the resource for other thread use
            lock.unlock();
        }
    }

    //Method to free resources
    public void close() {
    	running = false;
        lock.lock();
        try {
            if (socket != null && !socket.isClosed()) {
                socket.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            lock.unlock();
        }
    }

    // Set the listener object (FileUploadGUI in our case)
    public void setListener(ServerResultListener listener) {
        this.listener = listener;
    }
    
    //This method is user to initialize manually the socket, if we don't want automatic initialisation on constructor
    public void initSocket(String serverAddress, int port) throws UnknownHostException, IOException {
    	this.socket = new Socket(serverAddress, port);
        this.out = socket.getOutputStream();
        this.in = socket.getInputStream();
    }
}
